<?php
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

use Binali\Models\User;
use Binali\Config\Database;
use PDO;
use PDOException;
use Exception;
use PHPMailer\PHPMailer\PHPMailer;


require_once __DIR__ . '/../services/airtime.service.php';
require_once __DIR__ . '/../services/data.service.php';
require_once __DIR__ . '/../services/cable.service.php';
require_once __DIR__ . '/../services/electricity.service.php';
require_once __DIR__ . '/../services/exam.service.php';
require_once __DIR__ . '/../services/recharge.service.php';
require_once __DIR__ . '/../services/rechargepin.service.php';
require_once __DIR__ . '/../services/datapin.service.php';
require_once __DIR__ . '/../services/user.service.php';

$uri = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
$requestMethod = $_SERVER['REQUEST_METHOD'];

// Debug logging
error_log("Original Request URI: " . $uri);
error_log("Request Method: " . $requestMethod);

// Extract the endpoint from the URI
$uriParts = explode('/api/', $uri, 2);
$endpoint = isset($uriParts[1]) ? trim($uriParts[1], '/') : '';

// Debug logging
error_log("Extracted endpoint: " . $endpoint);

// Split for additional parameters if needed
$uriSegments = explode('/', $endpoint);

// Use the first segment as the endpoint
$endpoint = $uriSegments[0] ?? '';
$id = $uriSegments[1] ?? null;

// Debug logging
error_log("Processing endpoint: " . $endpoint);

// Initialize response array
$response = [
    'status' => 'error',
    'message' => 'Invalid endpoint',
    'data' => null
];

try {
    switch ($endpoint) {
        case 'delete-account':
            if ($requestMethod !== 'POST') {
                throw new Exception('Method not allowed');
            }

            $data = json_decode(file_get_contents("php://input"));
            if (!isset($data->userId) || !isset($data->reason)) {
                throw new Exception('Missing required parameters');
            }

            try {
                $user_service = new UserService();
                $response['data'] = $user_service->deleteAccount($data->userId, $data->reason);
                $response['status'] = 'success';
                $response['message'] = 'Account deleted successfully';
            } catch (PDOException $e) {
                error_log("Database error in delete-account: " . $e->getMessage());
                http_response_code(503);
                throw new Exception("Database service is currently unavailable. Please try again later.");
            } catch (Exception $e) {
                error_log("Error in delete-account: " . $e->getMessage());
                throw $e;
            }
            break;
        case 'airtime':
            if ($requestMethod !== 'POST') {
                throw new Exception('Method not allowed');
            }

            $rawInput = file_get_contents("php://input");
            error_log("Raw input: " . $rawInput);

            $data = json_decode($rawInput);
            if (json_last_error() !== JSON_ERROR_NONE) {
                throw new Exception('Invalid JSON: ' . json_last_error_msg());
            }

            error_log("Decoded data: " . print_r($data, true));

            if (!isset($data->network) || !isset($data->phone) || !isset($data->amount) || !isset($data->user_id)) {
                throw new Exception('Missing required parameters: ' .
                    (!isset($data->network) ? 'network ' : '') .
                    (!isset($data->phone) ? 'phone ' : '') .
                    (!isset($data->amount) ? 'amount ' : '') .
                    (!isset($data->user_id) ? 'user_id' : ''));
            }

            $service = new AirtimeService();
            $result = $service->purchaseAirtime($data->network, $data->phone, $data->amount, $data->user_id);
            $response['data'] = $result;
            $response['status'] = 'success';
            $response['message'] = 'Airtime purchase successful';
            break;

        case 'airtime-plans':
            $service = new AirtimeService();
            $response['data'] = $service->getAirtimePlans();
            $response['status'] = 'success';
            $response['message'] = 'Airtime plans fetched successfully';
            break;

        case 'data-plans':
            $service = new DataService();
            $networkId = isset($_GET['network']) ? $_GET['network'] : null;
            $dataType = isset($_GET['type']) ? $_GET['type'] : null;
            // Allow client to pass user id so service can select pricing (userprice/vendorprice)
            $userId = isset($_GET['user_id']) ? $_GET['user_id'] : (isset($_GET['userId']) ? $_GET['userId'] : null);
            $response['data'] = $service->getDataPlans($networkId, $dataType, $userId);
            $response['status'] = 'success';
            $response['message'] = 'Data plans fetched successfully';
            break;

        case 'purchase-data':
            if ($requestMethod !== 'POST') {
                throw new Exception('Method not allowed');
            }

            $data = json_decode(file_get_contents("php://input"));

            // Debug the received data
            error_log("Received data purchase request with data: " . print_r($data, true));

            // Map the incoming parameters to our internal format
            $networkId = $data->network ?? $data->network_id ?? $data->networkId ?? null;
            $phone = $data->mobile_number ?? $data->phone ?? $data->phoneNumber ?? null;
            $planId = $data->plan ?? $data->plan_id ?? $data->planId ?? null;
            $userId = $data->user_id ?? $data->userId ?? null;

            error_log("Mapped parameters:");
            error_log("Network ID: " . $networkId);
            error_log("Phone: " . $phone);
            error_log("Plan ID: " . $planId);
            error_log("User ID: " . $userId);

            if (!$networkId || !$phone || !$planId || !$userId) {
                error_log("Missing parameters. Required: network/network_id, mobile_number/phone, plan/plan_id, user_id");
                error_log("Received raw data: " . print_r($data, true));
                error_log("Mapped values: " . json_encode([
                    'network_id' => $networkId,
                    'phone' => $phone,
                    'plan_id' => $planId,
                    'user_id' => $userId
                ]));
                throw new Exception('Missing required parameters');
            }

            $service = new DataService();
            // Let the service return the structured result with status/message/data
            $svcResult = $service->purchaseData($networkId, $phone, $planId, $userId);

            // Normalize and propagate to top-level response
            $response['data'] = $svcResult['data'] ?? $svcResult;
            $response['message'] = $svcResult['message'] ?? '';
            $response['status'] = $svcResult['status'] ?? 'failed';

            // Set HTTP response code based on service status
            if ($response['status'] === 'failed') {
                http_response_code(500);
            } else {
                // success or processing
                http_response_code(200);
            }
            break;

        case 'cable-plans':
            $service = new CableService();
            $providerId = isset($_GET['provider']) ? $_GET['provider'] : null;
            $response['data'] = $service->getCablePlans($providerId);
            $response['status'] = 'success';
            $response['message'] = 'Cable plans fetched successfully';
            break;

        case 'electricity-providers':
            $service = new ElectricityService();
            $response['data'] = $service->getElectricityProviders();
            $response['status'] = 'success';
            $response['message'] = 'Electricity providers fetched successfully';
            break;

        case 'validate-meter':
            if ($requestMethod !== 'POST') {
                throw new Exception("Invalid request method");
            }

            $data = json_decode(file_get_contents("php://input"));
            if (!isset($data->meterNumber) || !isset($data->providerId)) {
                throw new Exception("Missing required parameters");
            }

            $service = new ElectricityService();
            $response['data'] = $service->validateMeterNumber(
                $data->meterNumber,
                $data->providerId,
                $data->meterType ?? 'prepaid'
            );
            $response['status'] = 'success';
            $response['message'] = 'Meter number validated successfully';
            break;

        case 'purchase-electricity':
            if ($requestMethod !== 'POST') {
                throw new Exception("Invalid request method");
            }

            $data = json_decode(file_get_contents("php://input"));
            if (!isset($data->meterNumber) || !isset($data->providerId) || !isset($data->amount)) {
                throw new Exception("Missing required parameters");
            }

            $service = new ElectricityService();
            $response['data'] = $service->purchaseElectricity(
                $data->meterNumber,
                $data->providerId,
                $data->amount,
                $data->meterType ?? 'prepaid',
                $data->phone ?? ''
            );
            $response['status'] = 'success';
            $response['message'] = 'Electricity purchase successful';
            break;

        case 'validate-iuc':
            if ($requestMethod !== 'POST') {
                throw new Exception('Method not allowed');
            }

            $data = json_decode(file_get_contents("php://input"));
            if (!isset($data->iucNumber) || !isset($data->providerId)) {
                throw new Exception('Missing required parameters');
            }

            $service = new CableService();
            $response['data'] = $service->validateIUCNumber($data->iucNumber, $data->providerId);
            $response['status'] = 'success';
            $response['message'] = 'IUC number validated successfully';
            break;

        case 'cable-subscription':
            if ($requestMethod !== 'POST') {
                throw new Exception('Method not allowed');
            }

            $data = json_decode(file_get_contents("php://input"));
            if (
                !isset($data->providerId) || !isset($data->planId) ||
                !isset($data->iucNumber) || !isset($data->phoneNumber) ||
                !isset($data->amount) || !isset($data->pin)
            ) {
                throw new Exception('Missing required parameters');
            }

            $service = new CableService();
            $response['data'] = $service->processCableSubscription(
                $data->providerId,
                $data->planId,
                $data->iucNumber,
                $data->phoneNumber,
                $data->amount,
                $data->pin
            );
            $response['status'] = 'success';
            $response['message'] = 'Cable subscription processed successfully';
            break;

        case 'exam-providers':
            $service = new ExamPinService();
            $result = $service->getExamProviders();
            $response['data'] = $result['data'];
            $response['status'] = $result['status'];
            $response['message'] = $result['message'];
            break;

        case 'exam-purchase':
            if ($requestMethod !== 'POST') {
                throw new Exception('Method not allowed');
            }

            $data = json_decode(file_get_contents("php://input"));
            error_log("Exam purchase request data: " . print_r($data, true));

            if (!isset($data->examId) || !isset($data->quantity) || !isset($data->userId) || !isset($data->pin)) {
                throw new Exception('Missing required parameters: examId, quantity, userId, pin');
            }

            $service = new ExamPinService();
            $result = $service->purchaseExamPin($data->examId, $data->quantity, $data->userId);

            $response['status'] = $result['status'];
            $response['message'] = $result['message'];
            $response['data'] = $result['data'];
            break;

        case 'recharge-card-plans':
            $service = new RechargeCardService();
            $response['data'] = $service->getRechargeCardPlans();
            $response['status'] = 'success';
            $response['message'] = 'Recharge card plans fetched successfully';
            break;

        case 'recharge-pin-plans':
            $service = new RechargePinService();
            $networkId = isset($_GET['network']) ? $_GET['network'] : null;
            $response['data'] = $service->getAvailablePins($networkId);
            $response['status'] = 'success';
            $response['message'] = 'Recharge pin plans fetched successfully';
            break;

        case 'data-pin-plans':
            $service = new DataPinService();
            $networkId = isset($_GET['network']) ? $_GET['network'] : null;
            $type = isset($_GET['type']) ? $_GET['type'] : null;

            try {
                $result = $service->getDataPinPlans($networkId, $type);
                $response['status'] = $result['status'];
                $response['message'] = $result['message'];
                $response['data'] = $result['data'];
            } catch (Exception $e) {
                $response['status'] = 'error';
                $response['message'] = $e->getMessage();
                $response['data'] = null;
            }
            break;

        case 'purchase-data-pin':
            if ($requestMethod !== 'POST') {
                http_response_code(405);
                $response['message'] = 'Method not allowed';
                break;
            }

            $data = json_decode(file_get_contents("php://input"));
            error_log("Data pin purchase request data: " . print_r($data, true));

            // Validate required parameters
            if (!isset($data->plan) || !isset($data->quantity) || !isset($data->name_on_card) || !isset($data->userId)) {
                error_log("Missing required parameters. Received: " . json_encode($data));
                http_response_code(400);
                $response['status'] = 'error';
                $response['message'] = 'Missing required parameters. Required: plan, quantity, name_on_card, userId';
                break;
            }

            try {
                $service = new DataPinService();
                $result = $service->purchaseDataPin(
                    $data->plan,
                    $data->quantity,
                    $data->name_on_card,
                    $data->userId
                );

                $response['status'] = $result['status'];
                $response['message'] = $result['message'];
                $response['data'] = $result['data'];
            } catch (Exception $e) {
                error_log("Error in data pin purchase: " . $e->getMessage());
                http_response_code(500);
                $response['status'] = 'error';
                $response['message'] = $e->getMessage();
            }
            break;

        case 'generate-account':
            // New behavior: backend decides which bank to create for the user.
            if ($requestMethod !== 'POST') {
                throw new Exception('Method not allowed');
            }

            $data = json_decode(file_get_contents("php://input"));
            if (!isset($data->user_id)) {
                throw new Exception('Missing required parameters: user_id');
            }

            require_once __DIR__ . '/../config/database.php';
            require_once __DIR__ . '/../models/user.php';

            try {
                error_log("Processing account generation request for user_id: " . $data->user_id);
                $db = new Database();

                // Get user details
                $query = "SELECT sFname, sLname, sPhone, sEmail FROM subscribers WHERE sId = ?";
                error_log("Executing query with user_id: " . $data->user_id);
                $result = $db->query($query, [$data->user_id]);

                if (empty($result)) {
                    throw new Exception('User not found');
                }

                $user = $result[0];
                error_log("User data found: " . print_r($user, true));

                // If user already has any account (sBankNo or sSterlingBank), return it
                $checkQuery = "SELECT sBankNo, sSterlingBank, sBankName FROM subscribers WHERE sId = ?";
                $checkResult = $db->query($checkQuery, [$data->user_id]);

                if (!empty($checkResult)) {
                    $existing = $checkResult[0];
                    if (!empty($existing['sBankNo'])) {
                        $response['status'] = 'success';
                        $response['message'] = 'Account already exists';
                        $response['data'] = [
                            'account_number' => $existing['sBankNo'],
                            'bank_name' => $existing['sBankName'] ?? 'StroWallet'
                        ];
                        break;
                    }
                    if (!empty($existing['sSterlingBank'])) {
                        $response['status'] = 'success';
                        $response['message'] = 'Account already exists';
                        $response['data'] = [
                            'account_number' => $existing['sSterlingBank'],
                            'bank_name' => $existing['sBankName'] ?? 'Sterling Bank'
                        ];
                        break;
                    }
                }

                // Create virtual account (StroWallet / backend decides which bank)
                $userModel = new User($db);
                error_log("Attempting to create virtual account for user ID: " . $data->user_id);

                $accountCreated = $userModel->createVirtualAccount(
                    $data->user_id,
                    $user['sFname'],
                    $user['sLname'],
                    $user['sPhone'],
                    $user['sEmail']
                );

                if ($accountCreated) {
                    // Fetch the newly created account number and bank name
                    $fetchQuery = "SELECT sBankNo, sSterlingBank, sBankName FROM subscribers WHERE sId = ?";
                    $fetchResult = $db->query($fetchQuery, [$data->user_id]);

                    if (!empty($fetchResult)) {
                        $accountInfo = $fetchResult[0];
                        // Prefer sBankNo (StroWallet/Noma) if present, otherwise sSterlingBank
                        $acct = !empty($accountInfo['sBankNo']) ? $accountInfo['sBankNo'] : $accountInfo['sSterlingBank'];
                        $response['status'] = 'success';
                        $response['message'] = 'Account generated successfully';
                        $response['data'] = [
                            'account_number' => $acct,
                            'bank_name' => $accountInfo['sBankName'] ?? ''
                        ];
                    } else {
                        throw new Exception('Account created but unable to fetch details');
                    }
                } else {
                    throw new Exception('Failed to generate account. Please try again later.');
                }
            } catch (Exception $e) {
                error_log("Error in generate-account: " . $e->getMessage());
                throw $e;
            }
            break;

        case 'transactions':
            $userId = isset($_GET['user_id']) ? $_GET['user_id'] : null;
            if (!$userId) {
                throw new Exception('User ID is required');
            }

            $query = "SELECT * FROM transactions WHERE sId = ? ORDER BY date DESC";
            $db = new Database();
            $transactions = $db->query($query, [$userId]);

            $response['data'] = $transactions;
            $response['status'] = 'success';
            $response['message'] = 'Transactions fetched successfully';
            break;

        case 'manual-payments':
            // Fetch manual payment records. Optional query param: user_id
            if ($requestMethod !== 'GET') {
                http_response_code(405);
                throw new Exception('Method not allowed');
            }

            // Optional filter by user id (accept user_id or userId)
            $userId = isset($_GET['user_id']) ? $_GET['user_id'] : (isset($_GET['userId']) ? $_GET['userId'] : null);

            require_once __DIR__ . '/../config/database.php';

            try {
                $db = new Database();

                if ($userId) {
                    $query = "SELECT * FROM manualfunds WHERE sId = ? ORDER BY dPosted DESC";
                    $payments = $db->query($query, [$userId]);
                } else {
                    $query = "SELECT * FROM manualfunds ORDER BY dPosted DESC";
                    $payments = $db->query($query);
                }

                $response['data'] = $payments;
                $response['status'] = 'success';
                $response['message'] = 'Manual payments fetched successfully';
            } catch (PDOException $e) {
                error_log("Database error in manual-payments: " . $e->getMessage());
                http_response_code(503);
                $response['status'] = 'error';
                $response['message'] = 'Database service is currently unavailable. Please try again later.';
            } catch (Exception $e) {
                error_log("Error in manual-payments: " . $e->getMessage());
                http_response_code(500);
                $response['status'] = 'error';
                $response['message'] = $e->getMessage();
            }
            break;

        case 'manual-payment':
            // Fetch a single manual payment destination (preferred: status=1), optional ?bank filter
            if ($requestMethod !== 'GET') {
                http_response_code(405);
                throw new Exception('Method not allowed');
            }
            // Return account details stored in sitesettings table (single row)
            require_once __DIR__ . '/../config/database.php';

            try {
                $db = new Database();
                // sitesettings usually has one row (sId=1). Fetch latest just in case.
                $query = "SELECT accountname, accountno, bankname FROM sitesettings ORDER BY sId DESC LIMIT 1";
                $rows = $db->query($query);

                if (!empty($rows)) {
                    $settings = $rows[0];
                    $response['data'] = [
                        'account_name' => $settings['accountname'] ?? '',
                        'account_number' => $settings['accountno'] ?? '',
                        'bank_name' => $settings['bankname'] ?? '',
                    ];
                    $response['status'] = 'success';
                    $response['message'] = 'Manual payment settings fetched successfully';
                } else {
                    $response['data'] = null;
                    $response['status'] = 'error';
                    $response['message'] = 'Site settings not found';
                }
            } catch (PDOException $e) {
                error_log("Database error in manual-payment (sitesettings): " . $e->getMessage());
                http_response_code(503);
                $response['status'] = 'error';
                $response['message'] = 'Database service is currently unavailable. Please try again later.';
            } catch (Exception $e) {
                error_log("Error in manual-payment (sitesettings): " . $e->getMessage());
                http_response_code(500);
                $response['status'] = 'error';
                $response['message'] = $e->getMessage();
            }
            break;

        case 'send-manual-proof':
            // Accepts POST with JSON body: amount, bank, sender, optional account_number, account_name, bank_name, user_id
            if ($requestMethod !== 'POST') {
                http_response_code(405);
                $response['message'] = 'Method not allowed';
                break;
            }

            $rawInput = file_get_contents("php://input");
            error_log("send-manual-proof raw input: " . $rawInput);

            $data = json_decode($rawInput);
            if (json_last_error() !== JSON_ERROR_NONE) {
                http_response_code(400);
                $response['message'] = 'Invalid JSON';
                break;
            }

            if (!isset($data->amount) || !isset($data->bank) || !isset($data->sender)) {
                http_response_code(400);
                $response['message'] = 'Missing required parameters: amount, bank, sender';
                break;
            }

            $amount = $data->amount;
            $bank = $data->bank;
            $sender = $data->sender;
            $accountNumber = isset($data->account_number) ? $data->account_number : '';
            $accountName = isset($data->account_name) ? $data->account_name : '';
            $bankName = isset($data->bank_name) ? $data->bank_name : '';

            // Load PHPMailer
            require_once __DIR__ . '/../vendor/autoload.php';
            

            try {
                $mail = new \PHPMailer\PHPMailer\PHPMailer(true);

                // SMTP settings (reusing existing project settings)
                $mail->isSMTP();
                $mail->Host = 'mail.binalidata.com.ng';
                $mail->SMTPAuth = true;
                $mail->Username = 'no-reply@binalidata.com.ng';
                $mail->Password = ']xG28YL,APm-+xbx';

                // Use STARTTLS for port 587
                $mail->SMTPSecure = PHPMailer::ENCRYPTION_STARTTLS;
                $mail->Port = 587;

                $mail->setFrom('no-reply@binalidata.com.ng', 'Binali One Data');
                $mail->addAddress('Muhammadbinali1234@gmail.com');

                $mail->isHTML(true);
                $mail->Subject = 'Manual Payment Proof Submission';

                $body = "<h3>Manual Payment Proof</h3>";
                $body .= "<p><strong>Amount:</strong> ₦" . htmlspecialchars($amount) . "</p>";
                $body .= "<p><strong>Bank Provided:</strong> " . htmlspecialchars($bank) . "</p>";
                if (!empty($bankName)) $body .= "<p><strong>Bank Name:</strong> " . htmlspecialchars($bankName) . "</p>";
                if (!empty($accountNumber)) $body .= "<p><strong>Account Number:</strong> " . htmlspecialchars($accountNumber) . "</p>";
                if (!empty($accountName)) $body .= "<p><strong>Account Name:</strong> " . htmlspecialchars($accountName) . "</p>";
                $body .= "<p><strong>Sender:</strong> " . htmlspecialchars($sender) . "</p>";
                if (isset($data->user_id)) $body .= "<p><strong>User ID:</strong> " . htmlspecialchars($data->user_id) . "</p>";
                $body .= "<p>Posted at: " . date('Y-m-d H:i:s') . "</p>";

                $mail->Body = $body;
                $mail->AltBody = strip_tags(str_replace(['<br>', '<br/>', '<p>', '</p>'], "\n", $body));

                $mail->send();

                // Also insert the proof record into the manualfunds table
                try {
                    require_once __DIR__ . '/../config/database.php';
                    $db = new Database();
                    $conn = $db->getConnection();

                    // Build account field similar to previous dumps (name + number) if available
                    $accountField = trim((!empty($accountName) ? $accountName . ' ' : '') . (!empty($accountNumber) ? $accountNumber : $sender));
                    $methodField = !empty($bankName) ? $bankName : $bank;
                    $sId = isset($data->user_id) ? $data->user_id : '';
                    $statusVal = 0; // default pending
                    $postedAt = date('Y-m-d H:i:s');

                    $insertStmt = $conn->prepare("INSERT INTO manualfunds (sId, amount, account, method, status, dPosted) VALUES (?, ?, ?, ?, ?, ?)");
                    $insertStmt->execute([$sId, $amount, $accountField, $methodField, $statusVal, $postedAt]);
                    $insertId = $conn->lastInsertId();

                    $response['status'] = 'success';
                    $response['message'] = 'Proof email sent and record saved successfully';
                    $response['data'] = ['insert_id' => $insertId];
                } catch (Exception $e) {
                    error_log('Error inserting manual proof into DB: ' . $e->getMessage());
                    http_response_code(500);
                    $response['status'] = 'error';
                    $response['message'] = 'Email sent but failed to save record: ' . $e->getMessage();
                    $response['data'] = null;
                }
            } catch (Exception $e) {
                error_log('Error sending manual proof email: ' . $e->getMessage());
                http_response_code(500);
                $response['status'] = 'error';
                $response['message'] = 'Unable to send email: ' . $e->getMessage();
            }

            break;

        case 'account-details':
            $subscriberId = isset($_GET['id']) ? $_GET['id'] : null;
            if (!$subscriberId) {
                throw new Exception('Subscriber ID is required');
            }

            $query = "SELECT sId, sFname, sLname, sEmail, sPhone, sType, sWallet, sRefWallet,
                sBankNo, sSterlingBank, sBankName, sRolexBank, sFidelityBank, sAsfiyBank,
                s9PSBBank, sPayvesselBank, sPagaBank, sPalmpayBank, sRegStatus, sLastActivity,
                sAccountLimit
                FROM subscribers WHERE sId = ?";

            $db = new Database();
            $result = $db->query($query, [$subscriberId]);

            if (empty($result)) {
                throw new Exception('Account details not found');
            }

            $account = $result[0];
            error_log("Raw account data from database: " . print_r($account, true));

            $response['data'] = [
                'sId' => $account['sId'],
                'sFname' => $account['sFname'],
                'sLname' => $account['sLname'],
                'sEmail' => $account['sEmail'],
                'sPhone' => $account['sPhone'],
                'sType' => (int)$account['sType'],
                'sWallet' => (float)$account['sWallet'],
                'sRefWallet' => (float)$account['sRefWallet'],
                'sBankNo' => $account['sBankNo'],
                'sSterlingBank' => $account['sSterlingBank'],
                'sBankName' => $account['sBankName'],
                'sRolexBank' => $account['sRolexBank'],
                'sFidelityBank' => $account['sFidelityBank'],
                'sAsfiyBank' => $account['sAsfiyBank'],
                's9PSBBank' => $account['s9PSBBank'],
                'sPayvesselBank' => $account['sPayvesselBank'],
                'sPagaBank' => $account['sPagaBank'],
                'sPalmpayBank' => $account['sPalmpayBank'],
                'sAccountLimit' => $account['sAccountLimit']
            ];
            error_log("Formatted response data: " . json_encode($response['data'], JSON_PRETTY_PRINT));
            $response['status'] = 'success';
            $response['message'] = 'Account details fetched successfully';
            break;

        case 'subscriber':
            $subscriberId = isset($_GET['id']) ? $_GET['id'] : null;
            if (!$subscriberId) {
                throw new Exception('Subscriber ID is required');
            }

            $query = "SELECT sId, sFname, sLname, sEmail, sPhone, sType, sWallet, sRefWallet, 
                sBankNo, sSterlingBank, sBankName, sRegStatus, sLastActivity,
                sRolexBank, sFidelityBank, sAsfiyBank, s9PSBBank, sPayvesselBank, 
                sPagaBank, sPalmpayBank 
                FROM subscribers WHERE sId = ?";

            $db = new Database();
            $result = $db->query($query, [$subscriberId]);

            if (empty($result)) {
                throw new Exception('Subscriber not found');
            }

            $subscriber = $result[0];
            // Format the response
            $response['data'] = [
                'sId' => $subscriber['sId'],
                'sFname' => $subscriber['sFname'],
                'sLname' => $subscriber['sLname'],
                'sEmail' => $subscriber['sEmail'],
                'sPhone' => $subscriber['sPhone'],
                'sType' => (int)$subscriber['sType'],
                'sWallet' => (float)$subscriber['sWallet'],
                'sRefWallet' => (float)$subscriber['sRefWallet'],
                'sBankNo' => $subscriber['sBankNo'],
                'sSterlingBank' => $subscriber['sSterlingBank'],
                'sBankName' => $subscriber['sBankName'],
                'sRolexBank' => $subscriber['sRolexBank'],
                'sFidelityBank' => $subscriber['sFidelityBank'],
                'sAsfiyBank' => $subscriber['sAsfiyBank'],
                's9PSBBank' => $subscriber['s9PSBBank'],
                'sPayvesselBank' => $subscriber['sPayvesselBank'],
                'sPagaBank' => $subscriber['sPagaBank'],
                'sPalmpayBank' => $subscriber['sPalmpayBank'],
                'sRegStatus' => (int)$subscriber['sRegStatus'],
                'lastActivity' => $subscriber['sLastActivity']
            ];
            $response['status'] = 'success';
            $response['message'] = 'Subscriber details fetched successfully';
            break;

        default:
            error_log("No matching endpoint found for: " . $endpoint);
            http_response_code(404);
            $response['message'] = "Endpoint '/$endpoint' not found";
            break;
    }
} catch (Exception $e) {
    $response['status'] = 'error';
    $response['message'] = $e->getMessage();
    http_response_code(500);
}

echo json_encode($response);
