<?php
namespace Binali\Models;

use PDO;
use PDOException;
use Exception;
use Binali\Config\Config;

/**
 * User Class
 * @package Binali\Models
 */
class User
{
    private $conn;
    private $table_name = "subscribers";
    
    private function getConnection() {
        if (!$this->conn) {
            throw new Exception("Database connection not initialized");
        }
        return $this->conn instanceof PDO ? $this->conn : 
               (method_exists($this->conn, 'getPDO') ? $this->conn->getPDO() : 
               (method_exists($this->conn, 'getConnection') ? $this->conn->getConnection() : 
               $this->conn));
    }
    
    public $sId;
    public $sApiKey;
    public $sFname;
    public $sLname;
    public $sEmail;
    public $sPhone;
    public $sPass;
    public $sReferal;
    public $sRegStatus;
    public $sRegDate;
    public $sWallet;
    public $reset_token;
    public $reset_token_expiry;
    
    public function __construct($db) {
        if ($db instanceof PDO) {
            $this->conn = $db;
        } elseif (method_exists($db, 'getPDO')) {
            $this->conn = $db->getPDO();
        } elseif (method_exists($db, 'getConnection')) {
            $this->conn = $db->getConnection();
        } else {
            $this->conn = $db;
        }
    }

    /**
     * Compute the website-compatible password hash.
     * This matches the legacy logic used by the website: substr(sha1(md5($password)), 3, 10)
     */
    private function computeWebsiteHash(string $password): string {
        return substr(sha1(md5($password)), 3, 10);
    }
    
    public function createVirtualAccount($id, $fname, $lname, $phone, $email) {
        try {
            if (!$this->conn) {
                error_log("Database connection not initialized");
                return false;
            }
            
            // Load config class
            require_once __DIR__ . '/../config/config.php';
            
            // Instantiate Config class with fully qualified namespace
            $config = new Config($this->conn);
        
        $fullname = $fname . " " . $lname;
        // Generate a unique reference using timestamp and random string
        $timestamp = time();
        $randomStr = bin2hex(random_bytes(4));
        $accref = "BINALI" . $id . $timestamp . $randomStr;
        error_log("Generated unique account reference: " . $accref);
        
    // Note: we no longer require Monnify credentials here. StroWallet is used
    // to create virtual accounts. Make sure the StroWallet config is present.
        
        // Use Strowallet to create a virtual bank account (Nombank MFB)
        // Assumptions:
        // - Config class exposes getStrowalletConfig() returning ['api_key' => '', 'base_url' => '']
        // - Strowallet expects a POST to {base_url}/virtual_accounts (adjust if your installation differs)
        // - The preferred bank identifier for Nombank MFB will be sent as 'bank' => 'nombank' (adjust if API requires a numeric bank code)
        $strowalletConfig = $config->getStrowalletConfig();
        if (!$strowalletConfig || empty($strowalletConfig['api_key']) || empty($strowalletConfig['base_url'])) {
            error_log("Missing Strowallet configuration (api_key or base_url)");
            return false;
        }

        $strowalletApiKey = $strowalletConfig['api_key'];
        $strowalletBase = rtrim($strowalletConfig['base_url'], '/');
        // Use the bank-specific endpoint for Nombank (amucha) per StroWallet docs.
        // Alternatively use '/api/virtual-bank/new-customer/' which defaults to Nombank.
        $create_url = $strowalletBase . '/api/virtual-bank/amucha';

        // Build payload according to StroWallet docs
        $payload = [
            'public_key' => $strowalletApiKey,
            'email' => $email,
            'account_name' => $fullname,
            'phone' => $phone,
            'webhook_url' => 'https://binalionedata.com.ng/',
            'mode' => 'strowalletMode'
        ];

        $curl = curl_init();
        curl_setopt_array($curl, array(
            CURLOPT_URL => $create_url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => "",
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 60,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => "POST",
            CURLOPT_POSTFIELDS => json_encode($payload),
            CURLOPT_HTTPHEADER => array(
                "Content-Type: application/json"
            ),
        ));

        $response = curl_exec($curl);
        $curlError = curl_error($curl);
        $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        curl_close($curl);

        error_log("Strowallet API Response Code: " . $httpCode);
        error_log("Strowallet API Response: " . $response);

        if ($curlError) {
            error_log("Curl Error: " . $curlError);
            return false;
        }

        $value = json_decode($response, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            error_log("JSON decode error (strowallet): " . json_last_error_msg());
            return false;
        }

        error_log("Decoded strowallet response: " . print_r($value, true));

        // Strowallet success shape may differ; try to extract an account number and bank name robustly
        $accountNumber = null;
        $bankName = null;

        // Normalize candidate container: prefer data if present, otherwise use top-level
        if (isset($value['data']) && is_array($value['data'])) {
            $d = $value['data'];
        } elseif (is_array($value)) {
            $d = $value;
        } else {
            $d = [];
        }

        // Common field names in various shapes
        if (isset($d['account_number'])) {
            $accountNumber = $d['account_number'];
        } elseif (isset($d['account_no'])) {
            $accountNumber = $d['account_no'];
        } elseif (isset($d['virtual_account'])) {
            $accountNumber = $d['virtual_account'];
        } elseif (isset($d['accountNumber'])) {
            $accountNumber = $d['accountNumber'];
        }

        if (isset($d['bank_name'])) {
            $bankName = $d['bank_name'];
        } elseif (isset($d['bank'])) {
            $bankName = $d['bank'];
        } elseif (isset($d['bankName'])) {
            $bankName = $d['bankName'];
        }

        // Additional fallback: some responses use top-level 'account_number' or 'bank_name'
        if (empty($accountNumber) && isset($value['account_number'])) {
            $accountNumber = $value['account_number'];
        }
        if (empty($bankName) && isset($value['bank_name'])) {
            $bankName = $value['bank_name'];
        }

        if (empty($accountNumber)) {
            error_log("Strowallet did not return an account number");
            return false;
        }

        // If bank name wasn't provided by Strowallet, default to Noma bank
        if (empty($bankName)) {
            $bankName = 'Noma bank';
        }

        // Persist the returned account to the subscribers table
        try {
            $query = "UPDATE " . $this->table_name . " 
                     SET sBankName = :bn, sBankNo = :we, accountReference = :ref 
                     WHERE sId = :id";
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(':bn', $bankName, PDO::PARAM_STR);
            $stmt->bindParam(':we', $accountNumber, PDO::PARAM_STR);
            $stmt->bindParam(':ref', $accref, PDO::PARAM_STR);
            $stmt->bindParam(':id', $id, PDO::PARAM_INT);
            return $stmt->execute();
        } catch (PDOException $e) {
            error_log("Database error: " . $e->getMessage());
            return false;
        }
        } catch (Exception $e) {
            error_log("Error in createVirtualAccount: " . $e->getMessage());
            return false;
        }
    }
    
        public function create() {
            try {
                // Generate API key
                $this->sApiKey = bin2hex(random_bytes(32));

                $query = "INSERT INTO {$this->table_name} 
                (sApiKey, sFname, sLname, sEmail, sPhone, sPass, sReferal, sWallet, sRegStatus, sRegDate)
                VALUES
                (:sApiKey, :sFname, :sLname, :sEmail, :sPhone, :sPass, :sReferal, 0, 0, NOW())";

                $stmt = $this->conn->prepare($query);

                // Sanitize input
                $this->sFname = htmlspecialchars(strip_tags($this->sFname));
                $this->sLname = htmlspecialchars(strip_tags($this->sLname));
                $this->sEmail = htmlspecialchars(strip_tags($this->sEmail));
                $this->sPhone = htmlspecialchars(strip_tags($this->sPhone));
                $this->sReferal = htmlspecialchars(strip_tags($this->sReferal));

                // Hash the password using website-compatible hash
                $password_hash = $this->computeWebsiteHash($this->sPass);

                // Bind values
                $stmt->bindParam(":sApiKey", $this->sApiKey);
                $stmt->bindParam(":sFname", $this->sFname);
                $stmt->bindParam(":sLname", $this->sLname);
                $stmt->bindParam(":sEmail", $this->sEmail);
                $stmt->bindParam(":sPhone", $this->sPhone);
                $stmt->bindParam(":sPass", $password_hash);
                $stmt->bindParam(":sReferal", $this->sReferal);

                if($stmt->execute()) {
                    // Get the last inserted ID
                    $lastId = $this->conn->lastInsertId();

                    // Attempt to create virtual account but do not fail registration if it fails.
                    try {
                        $this->createVirtualAccount(
                            $lastId,
                            $this->sFname,
                            $this->sLname,
                            $this->sPhone,
                            $this->sEmail
                        );
                    } catch (Exception $e) {
                        // Log the error but continue; account can be generated later.
                        error_log("Non-fatal: createVirtualAccount failed for sId={$lastId}: " . $e->getMessage());
                    }

                    // Registration succeeded regardless of virtual account creation outcome
                    return true;
                }

                return false;
            } catch (Exception $e) {
                error_log("Error in create: " . $e->getMessage());
                return false;
            }
        }
    
        public function emailExists() {
        $query = "SELECT sId, sFname, sLname, sPass, sPhone, sWallet, sEmail, sRegStatus 
                FROM " . $this->table_name . "
                WHERE sEmail = ?
                LIMIT 0,1";
    
        $stmt = $this->conn->prepare($query);
        $this->sEmail = htmlspecialchars(strip_tags($this->sEmail));
        $stmt->bindParam(1, $this->sEmail);
        $stmt->execute();
        
        $num = $stmt->rowCount();
        
        if($num > 0) {
            $row = $stmt->fetch(PDO::FETCH_ASSOC);
            $this->sId = $row['sId'];
            $this->sFname = $row['sFname'];
            $this->sLname = $row['sLname'];
            $this->sPass = $row['sPass'];
            $this->sPhone = $row['sPhone'];
            $this->sWallet = $row['sWallet'];
            $this->sEmail = $row['sEmail'];
            $this->sRegStatus = $row['sRegStatus'];
            // Debug log
            error_log("Retrieved hash from DB: " . $row['sPass']);
            return true;
        }
        return false;
    }
    
    public function validatePassword($plainPassword) {
        // Debug log to check values
        error_log("Plain password: " . $plainPassword);
        error_log("Stored hash: " . $this->sPass);
    // Compare using website's hash function
    $computed = $this->computeWebsiteHash($plainPassword);
    error_log("Computed hash: " . $computed);
    return hash_equals($this->sPass, $computed);
    }
    
    public function update() {
        $query = "UPDATE " . $this->table_name . "
                SET
                    sPass = :sPass
                WHERE sEmail = :sEmail";
    
        $stmt = $this->conn->prepare($query);
        
        $this->sEmail = htmlspecialchars(strip_tags($this->sEmail));
    // Use website-compatible hash
    $password_hash = $this->computeWebsiteHash($this->sPass);
        
        $stmt->bindParam(':sPass', $password_hash);
        $stmt->bindParam(':sEmail', $this->sEmail);
        
        if($stmt->execute()) {
            return true;
        }
        return false;
    }

    public function updateResetToken() {
        $query = "UPDATE " . $this->table_name . "
                SET
                    reset_token = :reset_token,
                    reset_token_expiry = :reset_token_expiry
                WHERE sEmail = :sEmail";
    
        $stmt = $this->conn->prepare($query);
        
        $this->sEmail = htmlspecialchars(strip_tags($this->sEmail));
        $this->reset_token = htmlspecialchars(strip_tags($this->reset_token));
        
        $stmt->bindParam(':reset_token', $this->reset_token);
        $stmt->bindParam(':reset_token_expiry', $this->reset_token_expiry);
        $stmt->bindParam(':sEmail', $this->sEmail);
        
        if($stmt->execute()) {
            return true;
        }
        return false;
    }

    public function validateResetToken($token) {
        $query = "SELECT sId FROM " . $this->table_name . "
                WHERE reset_token = :token
                AND reset_token_expiry > NOW()
                LIMIT 0,1";
    
        $stmt = $this->conn->prepare($query);
        
        $token = htmlspecialchars(strip_tags($token));
        $stmt->bindParam(':token', $token);
        $stmt->execute();
        
        if($stmt->rowCount() > 0) {
            $row = $stmt->fetch(PDO::FETCH_ASSOC);
            $this->sId = $row['sId'];
            return true;
        }
        return false;
    }

    public function resetPassword() {
        $query = "UPDATE " . $this->table_name . "
                SET
                    sPass = :sPass,
                    reset_token = NULL,
                    reset_token_expiry = NULL
                WHERE sId = :sId";
    
        $stmt = $this->conn->prepare($query);
        
    // Use website-compatible hash for reset as well
    $password_hash = $this->computeWebsiteHash($this->sPass);
        
        $stmt->bindParam(':sPass', $password_hash);
        $stmt->bindParam(':sId', $this->sId);
        
        try {
            if($stmt->execute()) {
                return true;
            }
            return false;
        } catch (PDOException $e) {
            error_log("Database error: " . $e->getMessage());
            return false;
        }
    }
}
