<?php
use Binali\Models;

use PDO;
use PDOException;
use Exception;
use Binali\Config\Config;
require_once __DIR__ . '/../db/database.php';
require_once __DIR__ . '/../config/config.php';

class AirtimeService {
    private $db;
    private $config;
    private $networkNames = [
        1 => "MTN",
        2 => "GLO",
        3 => "9MOBILE",
        4 => "AIRTEL"
    ];

    public function __construct() {
        $this->db = new Database();
        $this->config = new Config($this->db);
    }

    private function getNetworkName($networkId) {
        return $this->networkNames[$networkId] ?? null;
    }

    public function getAirtimePlans() {
        try {
            $query = "SELECT a.*, n.networkName 
                     FROM airtime a 
                     JOIN networks n ON a.aNetwork = n.networkId";
            return $this->db->query($query);
        } catch (Exception $e) {
            throw new Exception("Error fetching airtime plans: " . $e->getMessage());
        }
    }

    public function getAirtimePinPlans() {
        try {
            $query = "SELECT ap.*, n.networkName 
                     FROM airtimepin ap 
                     JOIN networks n ON ap.aNetwork = n.networkId";
            return $this->db->query($query);
        } catch (Exception $e) {
            throw new Exception("Error fetching airtime pin plans: " . $e->getMessage());
        }
    }

    public function getAirtimeProviderDetails($networkId, $type) {
        try {
            // Debug log the parameters
            error_log("Getting provider details for networkId: " . $networkId . ", type: " . $type);
            
            // Map network IDs to config prefixes
            $networkPrefix = [
                1 => 'mtn',
                2 => 'glo',
                3 => '9mobile',
                4 => 'airtel'
            ];
            
            $prefix = $networkPrefix[$networkId] ?? null;
            if (!$prefix) {
                throw new Exception("Invalid network ID: " . $networkId);
            }
            
            $keyName = $prefix . ucfirst($type) . "Key";
            $providerName = $prefix . ucfirst($type) . "Provider";
            
            error_log("Looking for config names: keyName=" . $keyName . ", providerName=" . $providerName);

            // Fetch both configurations directly
            $query = "SELECT 
                        (SELECT value FROM apiconfigs WHERE name = ?) as apiKey,
                        (SELECT value FROM apiconfigs WHERE name = ?) as provider";
            
            $result = $this->db->query($query, [$keyName, $providerName]);
            error_log("Provider details result: " . json_encode($result));
            
            if (empty($result) || !isset($result[0]['apiKey']) || !isset($result[0]['provider'])) {
                throw new Exception("Provider configuration not found for network " . $networkPrefix[$networkId]);
            }
            
            return $result;
        } catch (Exception $e) {
            error_log("Error in getAirtimeProviderDetails: " . $e->getMessage());
            throw new Exception("Error fetching provider details: " . $e->getMessage());
        }
    }

    private function getNetworkId($network) {
        // Convert string network names to IDs
        $networkMap = [
            'MTN' => 1,
            'GLO' => 2,
            '9MOBILE' => 3,
            'AIRTEL' => 4
        ];
        
        if (is_numeric($network)) {
            return (int)$network;
        }
        
        return $networkMap[strtoupper($network)] ?? null;
    }

    public function purchaseAirtime($network, $phone, $amount, $userId, $airtimeType = 'VTU') {
        try {
            // Convert network name to ID if needed
            $networkId = $this->getNetworkId($network);
            if ($networkId === null) {
                throw new Exception("Invalid network provider");
            }

            // Debug log
            error_log("Starting airtime purchase with params: " . json_encode([
                'network' => $network,
                'networkId' => $networkId,
                'phone' => $phone,
                'amount' => $amount,
                'userId' => $userId,
                'airtimeType' => $airtimeType
            ]));

            // Validate user exists and has sufficient balance
            $query = "SELECT sWallet FROM subscribers WHERE sId = ?";
            error_log("Executing query: " . $query . " with userId: " . $userId);
            
            $user = $this->db->query($query, [$userId]);
            error_log("Query result: " . json_encode($user));
            
            if (empty($user)) {
                throw new Exception("User not found");
            }

            $wallet = $user[0]['sWallet'];
            error_log("User wallet balance: " . $wallet);
            
            if ($wallet < $amount) {
                throw new Exception("Insufficient balance");
            }

            // Create reference for tracking
            $reference = 'AIR' . time() . rand(1000, 9999);
            
            // Get provider details for the network based on airtime type
            $type = $airtimeType === 'VTU' ? 'Vtu' : 'Sharesell';
            $providerDetails = $this->getAirtimeProviderDetails($networkId, $type);
            if (empty($providerDetails)) {
                throw new Exception("Provider configuration not found for network ID: $networkId");
            }
            
            error_log("Provider details: " . json_encode($providerDetails));
            
            // Prepare API call
            $apiUrl = $providerDetails[0]['provider'];
            $apiKey = $providerDetails[0]['apiKey'];

            // Prepare API call using ALRAHUZDATA API
            $curl = curl_init();
            
            // Set up request to ALRAHUZDATA API
            curl_setopt_array($curl, [
                CURLOPT_URL => "https://alrahuzdata.com.ng/api/topup/",
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_CUSTOMREQUEST => "POST",
                CURLOPT_POSTFIELDS => json_encode([
                    "network" => $networkId,
                    "amount" => $amount,
                    "mobile_number" => $phone,
                    "Ported_number" => true,
                    "airtime_type" => $airtimeType
                ]),
                CURLOPT_HTTPHEADER => [
                    "Authorization: Token " . $apiKey,
                    "Content-Type: application/json"
                ]
            ]);

            $response = curl_exec($curl);
            $err = curl_error($curl);
            $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
            curl_close($curl);

            error_log("API Response Code: " . $httpCode);
            error_log("API Response: " . $response);
            error_log("API Error (if any): " . $err);

            if ($err) {
                throw new Exception("Server Connection Error: " . $err);
            }

            if ($httpCode >= 500) {
                throw new Exception("Server error occurred. Please try again later. Status: " . $httpCode);
            }

            $result = json_decode($response, true);
            
            // Create transaction record
            $oldBal = $wallet;
            $newBal = $wallet - $amount;
            
            $insertQuery = "INSERT INTO transactions (sId, transref, servicename, servicedesc, amount, status, oldbal, newbal, profit, date, api_response, api_response_log) 
                          VALUES (?, ?, 'airtime', ?, ?, ?, ?, ?, 0, NOW(), ?, ?)";
            
            // Get API Status based on ALRAHUZDATA response format
            $transactionStatus = 0;
            $responseMessage = "Transaction Failed, Please Try Again Later";

            if ($httpCode === 200 || $httpCode === 201) {
                if (isset($result['Status']) && strtolower($result['Status']) === 'successful') {
                    $transactionStatus = 1;
                    $responseMessage = "Airtime purchase successful";
                    
                    // Deduct from wallet on success
                    $updateQuery = "UPDATE subscribers SET sWallet = sWallet - ? WHERE sId = ?";
                    $this->db->query($updateQuery, [$amount, $userId]);
                } elseif (isset($result['Status']) && strtolower($result['Status']) === 'pending') {
                    $transactionStatus = 0;
                    $responseMessage = "Transaction is processing";
                    error_log("Airtime processing log: " . json_encode($result));
                } else {
                    $transactionStatus = 2;
                    $responseMessage = isset($result['message']) ? $result['message'] : "Transaction failed";
                    error_log("Airtime fail log: " . json_encode($result));
                }
            } else {
                $transactionStatus = 2;
                $responseMessage = "Transaction failed due to server error";
                error_log("Airtime fail log: " . json_encode($result));
            }
            
            // Calculate profit (if any)
            $profit = 0; // Set profit calculation logic here if needed
            
            // Insert transaction record - ensure all values are properly formatted
            $this->db->query($insertQuery, [
                $userId, 
                $reference,
                "Airtime purchase for $phone",
                (string)$amount, // Cast to string as amount is VARCHAR in DB
                (int)$transactionStatus, // Cast to int as status is tinyint in DB
                (string)$oldBal, // Cast to string as oldbal is VARCHAR in DB
                (string)$newBal, // Cast to string as newbal is VARCHAR in DB
                $response,
                "API Error: " . $err . "\nAPI Response: " . $response
            ]);

            $transIdQuery = "SELECT LAST_INSERT_ID() as id";
            $transResult = $this->db->query($transIdQuery);
            $transactionId = $transResult[0]['id'];

            // Generate BIN transaction ID
            $binTransactionId = 'BIN_' . strtoupper(substr(uniqid() . bin2hex(random_bytes(4)), 0, 15));
            
            if ($transactionStatus === 1) {
                return [
                    'status' => 'success',
                    'message' => $responseMessage,
                    'data' => [
                        'transactionId' => $binTransactionId,
                        'reference' => $reference,
                        'amount' => $amount,
                        'phone' => $phone,
                        'network' => $network,
                        'status' => 'successful'
                    ]
                ];
            } else if ($transactionStatus === 0) {
                return [
                    'status' => 'processing',
                    'message' => $responseMessage,
                    'data' => [
                        'transactionId' => $binTransactionId,
                        'reference' => $reference
                    ]
                ];
            } else {
                return [
                    'status' => 'failed',
                    'message' => $responseMessage,
                    'data' => [
                        'transactionId' => $binTransactionId,
                        'reference' => $reference,
                        'amount' => $amount,
                        'phone' => $phone,
                        'network' => $network,
                        'status' => 'failed'
                    ]
                ];
            }
        } catch (Exception $e) {
            $errorMessage = $e->getMessage();
            error_log("Airtime purchase error: " . $errorMessage);
            
            // Log the error
            error_log("Airtime error log: " . $errorMessage);
            
            // Create failed transaction record if we have the necessary info
            if (isset($userId) && isset($reference) && isset($amount) && isset($phone)) {
                try {
                    $insertQuery = "INSERT INTO transactions (sId, transref, servicename, servicedesc, amount, status, oldbal, newbal, profit, date, api_response_log) 
                                  VALUES (?, ?, 'airtime', ?, ?, 2, ?, ?, 0, NOW(), ?)";
                    $this->db->query($insertQuery, [
                        $userId,
                        $reference,
                        "Failed airtime purchase for $phone",
                        (string)$amount,
                        (string)($wallet ?? '0'),
                        (string)($wallet ?? '0'),
                        $errorMessage
                    ]);
                } catch (Exception $logError) {
                    error_log("Error logging failed transaction: " . $logError->getMessage());
                }
            }

            // Generate BIN transaction ID for failed transaction
            $binTransactionId = 'BIN_' . strtoupper(substr(uniqid() . bin2hex(random_bytes(4)), 0, 15));

            // Return error response
            return [
                'status' => 'failed',
                'message' => $errorMessage,
                'data' => [
                    'transactionId' => $binTransactionId,
                    'phone' => $phone ?? null,
                    'amount' => $amount ?? null,
                    'reference' => $reference ?? null,
                    'status' => 'failed'
                ]
            ];
        }
    }
}
?>
