<?php
require_once __DIR__ . '/../db/database.php';

class DataService {
    private $db;

    public function __construct() {
        $this->db = new Database();
    }

public function getDataPlans($networkId = null, $dataType = null, $userId = null) {
    try {
        // Debug log the input parameters
        error_log("Fetching data plans with networkId: " . $networkId . ", dataType: " . $dataType . ", userId: " . $userId);

        // Determine price column based on user type (default to userprice)
        $priceCol = 'price';
        if ($userId !== null) {
            $u = $this->db->query("SELECT sType FROM subscribers WHERE sId = ?", [$userId]);
            if (!empty($u) && isset($u[0]['sType'])) {
                $sType = intval($u[0]['sType']);
                $priceCol = ($sType === 2) ? 'vendorprice' : 'userprice';
            } else {
                // fallback
                $priceCol = 'userprice';
            }
        } else {
            // default listing price when user not provided
            $priceCol = 'price';
        }

        // Safe select by inserting column name directly into SQL string
        $query = "SELECT 
                    dp.pId as id,              -- Internal DB ID
                    dp.planid as planCode,     -- API provider plan code
                    dp.name, 
                    dp." . $priceCol . " as price, 
                    dp.type as planType, 
                    dp.datanetwork as networkId, 
                    dp.day as validity
                  FROM dataplans dp";

        $params = [];
        $conditions = [];

        if ($networkId) {
            $conditions[] = "dp.datanetwork = ?";
            $params[] = $networkId;
        }

        if ($dataType) {
            $conditions[] = "dp.type = ?";
            $params[] = $dataType;
        }

        if (!empty($conditions)) {
            $query .= " WHERE " . implode(" AND ", $conditions);
        }

        $query .= " ORDER BY CAST(dp." . $priceCol . " AS DECIMAL(10,2)) ASC";

        error_log("Executing query: " . $query . " with params: " . json_encode($params));

        $result = $this->db->query($query, $params);

        error_log("Query result: " . json_encode($result));

        return $result;

    } catch (Exception $e) {
        throw new Exception("Error fetching data plans: " . $e->getMessage());
    }
}


    public function getDataProviderDetails($networkId, $dataGroup) {
        try {
            // Ensure dataGroup is not null and has a value
            if (empty($dataGroup)) {
                throw new Exception("Data group cannot be empty");
            }
            
            // Standardize network ID format (ensure it's a valid number)
            $networkId = intval($networkId);
            if ($networkId <= 0 || $networkId > 4) {
                throw new Exception("Invalid network ID: $networkId");
            }

            // Convert dataGroup to the correct format (SME, Corporate, Gifting)
            // Handle common variations in case
            $dataGroup = strtolower(trim($dataGroup));
            if ($dataGroup === 'sme') {
                $formattedGroup = 'SME';
            } elseif ($dataGroup === 'corporate' || $dataGroup === 'corp') {
                $formattedGroup = 'Corporate';
            } elseif ($dataGroup === 'gifting') {
                $formattedGroup = 'Gifting';
            } else {
                throw new Exception("Invalid data group type: $dataGroup. Expected: SME, Corporate, or Gifting");
            }
            
            // Debug logs for troubleshooting
            error_log("Getting provider details for network: $networkId, group: $formattedGroup");
            error_log("Original data group: $dataGroup, Formatted group: $formattedGroup");
            
            // First, let's log the current configuration
            error_log("Checking dataapiconfigs table contents...");
            $checkQuery = "SELECT * FROM dataapiconfigs WHERE name LIKE ?";
            $pattern = strtolower($networkId) . "%";
            $configs = $this->db->query($checkQuery, [$pattern]);
            error_log("Found configs for network $networkId: " . json_encode($configs));

            // Get network name for better error messages
            $networkNames = ['1' => 'MTN', '2' => 'Airtel', '3' => 'Glo', '4' => '9mobile'];
            $networkName = $networkNames[strval($networkId)] ?? "Unknown";

            // Map network IDs to prefix names
            $networkPrefixes = [
                1 => 'mtn',
                2 => 'airtel',
                3 => 'glo',
                4 => '9mobile'
            ];
            
            $networkPrefix = $networkPrefixes[$networkId] ?? strtolower($networkId);
            
            // Build config names using database convention
            $keyName = $networkPrefix . $formattedGroup . "Api";
            $providerName = $networkPrefix . $formattedGroup . "Provider";
            
            // Debug log the exact names we're looking for
            error_log("Searching for configurations:");
            error_log("Key config name: $keyName");
            error_log("Provider config name: $providerName");
            
            // Get configurations from dataapiconfigs table
            $configQuery = "SELECT name, value FROM dataapiconfigs WHERE name IN (?, ?)";
            $configs = $this->db->query($configQuery, [$keyName, $providerName]);
            
            // Debug log all found configurations
            error_log("Found configurations: " . json_encode($configs));
            
            // Process results
            $apiKey = null;
            $provider = null;
            foreach ($configs as $config) {
                if ($config['name'] === $keyName) {
                    $apiKey = $config['value'];
                } else if ($config['name'] === $providerName) {
                    $provider = $config['value'];
                }
            }
            
            // Detailed error logging
            if (!$apiKey && !$provider) {
                error_log("ERROR: Both API key and provider configurations are missing");
                throw new Exception("No provider configuration found for $networkName $formattedGroup services");
            } else if (!$apiKey) {
                error_log("ERROR: API key configuration is missing");
                throw new Exception("API key not found for $networkName $formattedGroup services");
            } else if (!$provider) {
                error_log("ERROR: Provider configuration is missing");
                throw new Exception("Provider URL not found for $networkName $formattedGroup services");
            }
            
            return [[
                'apiKey' => $apiKey,
                'provider' => $provider
            ]];
        } catch (Exception $e) {
            throw new Exception("Error fetching provider details: " . $e->getMessage());
        }
    }

public function purchaseData($networkId, $phoneNumber, $planCode, $userId) {
    try {
        error_log("\n=== Data Purchase Debug Info ===");
        error_log("Input Parameters:");
        error_log("Network ID: " . $networkId);
        error_log("Phone Number: " . $phoneNumber);
        error_log("Plan Code: " . $planCode);
        error_log("User ID: " . $userId);

        // Determine user type and price column before starting transaction
        $userInfo = $this->db->query("SELECT sType, sWallet as balance FROM subscribers WHERE sId = ?", [$userId]);
        if (empty($userInfo)) {
            throw new Exception("User not found");
        }
        $sType = intval($userInfo[0]['sType']);
        $priceCol = ($sType === 2) ? 'vendorprice' : 'userprice';
        error_log("User sType: " . $sType . ", using price column: " . $priceCol);

        // Start transaction
        if (!$this->db->beginTransaction()) {
            error_log("Transaction start failed!");
            throw new Exception("Could not start transaction");
        }

        // Get plan details using API planCode with the correct price column
        $planQuery = "SELECT pId, name, " . $priceCol . " as price, type as planType, datanetwork, day, planid as planCode 
                      FROM dataplans 
                      WHERE planid = ?";
        error_log("\nQuerying Database:");
        error_log("SQL Query: " . $planQuery);
        error_log("Parameters: " . json_encode([strval($planCode)]));
        
        $plan = $this->db->query($planQuery, [strval($planCode)]);
        error_log("Query Result: " . json_encode($plan));
        
        if (empty($plan)) {
            error_log("No plan found in database with planid: " . $planCode);
            // Let's check what plans actually exist
            $allPlans = $this->db->query("SELECT planid FROM dataplans LIMIT 10");
            error_log("Available planids (first 10): " . json_encode($allPlans));
            $this->db->rollBack();
            throw new Exception("Invalid plan selected");
        }
        $plan = $plan[0];
        error_log("\nSelected Plan Details:");
        error_log(json_encode($plan, JSON_PRETTY_PRINT));

        // Verify user balance (we already fetched balance earlier)
        $currentBalance = floatval($userInfo[0]['balance']);
        $planPrice = floatval($plan['price']);

        if ($currentBalance < $planPrice) {
            throw new Exception("Insufficient balance");
        }

        // Deduct amount from user balance
        $updateBalanceQuery = "UPDATE subscribers SET sWallet = sWallet - ? WHERE sId = ?";
        $this->db->query($updateBalanceQuery, [$planPrice, $userId]);

        // Create transaction record
        $reference = 'DATA_' . time() . '_' . rand(1000, 9999);
        $description = $plan['name'] . ' Data Purchase for ' . $phoneNumber;
        $newBalance = $currentBalance - $planPrice;
        $apiLog = json_encode([
            'network' => $networkId,
            'phone' => $phoneNumber,
            'plan' => $plan['name'],
            'planId' => $plan['planCode'],
            'type' => $plan['planType']
        ]);

        $transactionQuery = "INSERT INTO transactions 
            (sId, servicename, servicedesc, amount, status, oldbal, newbal, transref, api_response_log) 
            VALUES (?, 'data', ?, ?, 0, ?, ?, ?, ?)";
        $this->db->query($transactionQuery, [
            $userId,
            $description,
            $planPrice,
            $currentBalance,
            $newBalance,
            $reference,
            $apiLog
        ]);
        $transactionId = $this->db->lastInsertId();

        // Get provider details for the network and plan type
        $providerDetails = $this->getDataProviderDetails($networkId, $plan['planType']);
        if (empty($providerDetails)) {
            throw new Exception("Could not get provider details");
        }
        $providerConfig = $providerDetails[0];
        
        error_log("\nAPI Request Details:");
        error_log("Provider URL: " . $providerConfig['provider']);
        error_log("Network ID: " . $networkId);
        error_log("Phone: " . $phoneNumber);
        error_log("Plan Code: " . $plan['planCode']);

        // Prepare the exact request payload
        $requestBody = [
            'network' => intval($networkId),
            'mobile_number' => $phoneNumber,
            'plan' => intval($planCode),  // Use the original planCode parameter
            'Ported_number' => true
        ];
        error_log("Exact Request Body being sent: " . json_encode($requestBody, JSON_PRETTY_PRINT));
        
        // API request to provider
        $curl = curl_init();
        curl_setopt_array($curl, array(
            CURLOPT_URL => $providerConfig['provider'],
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => '',
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 0,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => 'POST',
            CURLOPT_SSL_VERIFYPEER => false,  // Added to handle HTTPS
            CURLOPT_SSL_VERIFYHOST => false,  // Added to handle HTTPS
            CURLOPT_POSTFIELDS => json_encode($requestBody),
            CURLOPT_HTTPHEADER => array(
                'Authorization: Token ' . $providerConfig['apiKey'],
                'Content-Type: application/json',
                'Accept: application/json'
            ),
        ));


        $response = curl_exec($curl);
        $err = curl_error($curl);
        $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        curl_close($curl);

        error_log("\nAPI Response:");
        error_log("HTTP Code: " . $httpCode);
        error_log("Response: " . $response);
        error_log("Curl Error: " . ($err ?: 'None'));

        if ($err) {
            throw new Exception("API Call Failed: " . $err);
        }

        $result = json_decode($response);
        
        if (!$result) {
            error_log("Failed to decode JSON response");
            throw new Exception("Invalid API response format");
        }
        
        error_log("Decoded Response: " . json_encode($result, JSON_PRETTY_PRINT));

        // Update API response
        $updateApiResponseQuery = "UPDATE transactions SET api_response = ? WHERE tId = ?";
        $this->db->query($updateApiResponseQuery, [$response, $transactionId]);

        // Check API status (robust parsing to handle different provider response shapes)
        $status = 0; // 0 = processing, 1 = success, 2 = failed
        $message = '';

        // Helper: read various fields that providers might return
        $providerStatus = null;
        $providerMessage = null;

        if (isset($result->status)) {
            $providerStatus = strtolower((string)$result->status);
        } elseif (isset($result->Status)) {
            $providerStatus = strtolower((string)$result->Status);
        } elseif (isset($result->data->status)) {
            $providerStatus = strtolower((string)$result->data->status);
        } elseif (isset($result->data->Status)) {
            $providerStatus = strtolower((string)$result->data->Status);
        }

        if (isset($result->message)) {
            $providerMessage = $result->message;
        } elseif (isset($result->msg)) {
            $providerMessage = $result->msg;
        } elseif (isset($result->api_response)) {
            $providerMessage = $result->api_response;
        }

        // Normalize status detection
        if ($providerStatus !== null) {
            if (in_array($providerStatus, ['success', 'successful', 'ok', 'completed', 'true'], true)) {
                $status = 1;
            } elseif (in_array($providerStatus, ['failed', 'error', 'failed_transaction', 'false'], true)) {
                $status = 2;
            } else {
                $status = 0; // unknown -> processing
            }
        } else {
            // No explicit status field: try to infer from known keys
            if (!empty($result->Status) || !empty($result->status)) {
                $s = strtolower((string)($result->Status ?? $result->status));
                if (strpos($s, 'success') !== false) {
                    $status = 1;
                } elseif (strpos($s, 'fail') !== false || strpos($s, 'error') !== false) {
                    $status = 2;
                }
            }
        }

        // If provider explicitly returned a failure, refund the user
        if ($status === 2) {
            $message = $providerMessage ?? ($result->message ?? 'Transaction failed');
            $this->db->query("UPDATE subscribers SET sWallet = sWallet + ? WHERE sId = ?", [$planPrice, $userId]);
        } else {
            // success or processing
            $message = $providerMessage ?? ($result->message ?? 'Data purchase ' . ($status === 1 ? 'successful' : 'processing'));
        }

        // Final transaction status update
        $this->db->query("UPDATE transactions SET status = ?, api_response = ? WHERE tId = ?", [$status, $response, $transactionId]);
        $this->db->commit();

        return [
            'status' => $status === 1 ? 'success' : ($status === 0 ? 'processing' : 'failed'),
            'message' => $message,
            'data' => [
                'transactionId' => 'BIN_' . strtoupper(uniqid()),
                'reference' => $reference,
                'amount' => $planPrice,
                'phone' => $phoneNumber,
                'network' => $networkId,
                'status' => $status === 1 ? 'success' : ($status === 0 ? 'processing' : 'failed')
            ]
        ];

    } catch (Exception $e) {
        if (isset($transactionId)) {
            $this->db->query("UPDATE transactions SET status = 2, api_response = ? WHERE tId = ?", [json_encode(['error' => $e->getMessage()]), $transactionId]);
            $this->db->query("UPDATE subscribers SET sWallet = sWallet + ? WHERE sId = ?", [$planPrice ?? 0, $userId]);
            $this->db->commit();
        } elseif ($this->db->inTransaction()) {
            $this->db->rollBack();
        }

        return [
            'status' => 'failed',
            'message' => $e->getMessage(),
            'data' => [
                'transactionId' => 'BIN_' . strtoupper(uniqid()),
                'reference' => $reference ?? null,
                'amount' => $planPrice ?? null,
                'phone' => $phoneNumber,
                'network' => $networkId,
                'status' => 'failed'
            ]
        ];
    }
}
}
?>
