<?php
require_once __DIR__ . '/../db/database.php';

class ElectricityService {
    private $db;

    public function __construct() {
        $this->db = new Database();
    }

    public function getElectricityProviders() {
        try {
            $query = "SELECT eId, provider, abbreviation FROM electricityid WHERE providerStatus = 'On' ORDER BY provider ASC";
            return $this->db->query($query);
        } catch (Exception $e) {
            throw new Exception("Error fetching electricity providers: " . $e->getMessage());
        }
    }

    public function getProviderDetails() {
        try {
            // Get all required API configurations in a single query
            $query = "SELECT 
                        (SELECT value FROM apiconfigs WHERE name = 'meterProvider') as provider,
                        (SELECT value FROM apiconfigs WHERE name = 'meterApi') as apiKey,
                        (SELECT value FROM apiconfigs WHERE name = 'meterVerificationProvider') as verificationProvider,
                        (SELECT value FROM apiconfigs WHERE name = 'meterVerificationApi') as verificationKey,
                        (SELECT value FROM apiconfigs WHERE name = 'meterProvider') as purchaseProvider,
                        (SELECT value FROM apiconfigs WHERE name = 'meterApi') as purchaseKey";
            
            $result = $this->db->query($query);
            
            if (empty($result)) {
                throw new Exception("No API configuration found");
            }
            
            return $result;
        } catch (Exception $e) {
            throw new Exception("Error fetching provider details: " . $e->getMessage());
        }
    }

    public function purchaseElectricity($meterNumber, $providerId, $amount, $meterType = 'prepaid', $phone = '') {
        try {
            // Get the provider information
            $query = "SELECT provider, abbreviation FROM electricityid WHERE eId = ?";
            $provider = $this->db->query($query, [$providerId]);
            
            if (empty($provider)) {
                throw new Exception("Invalid provider ID");
            }

            $provider = $provider[0];
            
            // Get API configuration
            $apiDetails = $this->getProviderDetails()[0];
            
            // Get base URL and API key from configuration
            $baseUrl = $apiDetails['provider'];
            if (empty($baseUrl)) {
                throw new Exception("Purchase provider URL not configured");
            }
            
            $apiKey = $apiDetails['apiKey'];
            if (empty($apiKey)) {
                throw new Exception("Purchase API key not configured");
            }

            // Build URL with query parameters
            $queryParams = http_build_query([
                'meternumber' => $meterNumber,
                'disconame' => $provider['abbreviation'],
                'mtype' => strtolower($meterType),
                'amount' => $amount,
                'phone' => $phone
            ]);
            $url = $baseUrl . '?' . $queryParams;

            $curl = curl_init();
            curl_setopt_array($curl, array(
                CURLOPT_URL => $url,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_ENCODING => '',
                CURLOPT_MAXREDIRS => 10,
                CURLOPT_TIMEOUT => 30,
                CURLOPT_FOLLOWLOCATION => true,
                CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
                CURLOPT_CUSTOMREQUEST => 'POST',
                CURLOPT_HTTPHEADER => array(
                    'Authorization: Token ' . $apiKey,
                    'Content-Type: application/json'
                ),
            ));

            $response = curl_exec($curl);
            $err = curl_error($curl);
            curl_close($curl);

            if ($err) {
                throw new Exception("cURL Error: " . $err);
            }

            $result = json_decode($response, true);

            if ($result) {
                return array(
                    'status' => 'success',
                    'data' => array(
                        'token' => $result['token'] ?? null,
                        'units' => $result['units'] ?? null,
                        'amount' => $result['amount'] ?? $amount,
                        'meter_number' => $meterNumber,
                        'provider' => $provider['provider'],
                        'reference' => $result['reference'] ?? null,
                        'message' => $result['message'] ?? 'Purchase successful'
                    )
                );
            } else {
                return array(
                    'status' => 'error',
                    'message' => 'Failed to process electricity purchase'
                );
            }

        } catch (Exception $e) {
            throw new Exception("Error purchasing electricity: " . $e->getMessage());
        }
    }

    public function validateMeterNumber($meterNumber, $providerId, $meterType = 'prepaid') {
        try {
            // Get the provider information
            $query = "SELECT provider, abbreviation, electricityid FROM electricityid WHERE eId = ?";
            $provider = $this->db->query($query, [$providerId]);
            
            if (empty($provider)) {
                throw new Exception("Invalid provider ID");
            }

            $provider = $provider[0];
            
            // Get API configuration
            $apiDetails = $this->getProviderDetails()[0];
            
            // Convert meter type to lowercase for consistency
            $meterType = strtolower($meterType);
            
            // Prepare the API request
            $curl = curl_init();
            
            // Get base URL and API key from configuration
            $baseUrl = $apiDetails['verificationProvider'];
            if (empty($baseUrl)) {
                throw new Exception("Verification provider URL not configured");
            }
            
            $apiKey = $apiDetails['verificationKey'];
            if (empty($apiKey)) {
                throw new Exception("Verification API key not configured");
            }

            // Build URL with query parameters
            $queryParams = http_build_query([
                'meternumber' => $meterNumber,
                'disconame' => $provider['electricityid'],
                'mtype' => $meterType // Use the parameter passed to the function
            ]);
            $url = $baseUrl . '?' . $queryParams;

            // Log the request details
            error_log("Meter Validation Request:");
            error_log("URL: " . $url);
            error_log("Provider: " . $provider['provider']);
            error_log("Abbreviation: " . $provider['abbreviation']);
            error_log("Meter Type: " . $meterType);
            error_log("API Key: " . substr($apiKey, 0, 10) . '...');

            curl_setopt_array($curl, array(
                CURLOPT_URL => $url,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_ENCODING => '',
                CURLOPT_MAXREDIRS => 10,
                CURLOPT_TIMEOUT => 30,
                CURLOPT_FOLLOWLOCATION => true,
                CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
                CURLOPT_CUSTOMREQUEST => 'GET',
                CURLOPT_HTTPHEADER => array(
                    'Authorization: Token ' . $apiKey,
                    'Content-Type: application/json'
                ),
            ));

            $response = curl_exec($curl);
            $err = curl_error($curl);
            $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
            curl_close($curl);

            // Log the response
            error_log("Meter Validation Response:");
            error_log("HTTP Status Code: " . $httpCode);
            error_log("Raw Response: " . $response);
            if ($err) {
                error_log("cURL Error: " . $err);
                throw new Exception("cURL Error: " . $err);
            }

            $result = json_decode($response, true);
            error_log("Decoded Response: " . print_r($result, true));

            // Check if we got a valid response
            if (!$result) {
                return array(
                    'status' => 'error',
                    'message' => 'Failed to get response from validation service'
                );
            }

            // Check if the meter is invalid or if required data is missing
            $isInvalid = isset($result['invalid']) && $result['invalid'] === true;
            $hasRequiredData = !empty($result['name']) && !empty($result['address']);

            if ($isInvalid || !$hasRequiredData) {
                return array(
                    'status' => 'error',
                    'message' => $isInvalid ? 'Invalid meter number' : 'Incomplete meter information received'
                );
            }

            // If we reach here, we have a valid meter with complete information
            return array(
                'status' => 'success',
                'data' => array(
                    'invalid' => false,
                    'name' => $result['name'],
                    'address' => $result['address'],
                    'meter_number' => $meterNumber,
                    'provider' => $provider['provider']
                )
            );

        } catch (Exception $e) {
            throw new Exception("Error validating meter number: " . $e->getMessage());
        }
    }
}
?>
