<?php
require_once __DIR__ . '/../config/database.php';

use Binali\Config\Database;
use PDO;
use Exception;

class UserService {
    private $conn;
    private $database;

    public function __construct() {
        try {
            $this->database = new Database();
            $this->initializeConnection();
        } catch (Exception $e) {
            error_log("Database connection error in UserService constructor: " . $e->getMessage());
            throw new Exception("Database connection failed. Please try again later.");
        }
    }

    private function initializeConnection() {
        try {
            // Get a new connection using the method from Database class
            $this->conn = $this->database->getConnection();
            
            if (!$this->conn || !($this->conn instanceof PDO)) {
                error_log("Invalid database connection object received");
                throw new Exception("Database connection is not valid");
            }
            
            // Test the connection
            $this->conn->query("SELECT 1");
            error_log("Database connection successfully initialized");
            
        } catch (PDOException $e) {
            error_log("Database connection test failed: " . $e->getMessage());
            throw new Exception("Database connection is not active");
        }
    }

    private function ensureAccountDeletionsTableExists() {
        try {
            $createTableQuery = "CREATE TABLE IF NOT EXISTS account_deletions (
                id INT PRIMARY KEY AUTO_INCREMENT,
                user_id INT NOT NULL,
                reason TEXT NOT NULL,
                deletion_date DATETIME NOT NULL,
                FOREIGN KEY (user_id) REFERENCES subscribers(sId)
            )";
            $this->conn->exec($createTableQuery);
        } catch (Exception $e) {
            error_log("Error creating account_deletions table: " . $e->getMessage());
            // Don't throw, as the table might already exist
        }
    }

    public function deleteAccount($userId, $reason) {
        try {
            error_log("Attempting to delete account for user ID: " . $userId);
            
            // Ensure we have a valid connection
            if (!$this->database->isConnected()) {
                error_log("Database connection lost, reinitializing...");
                $this->initializeConnection();
            }
            
            if (!$this->conn) {
                throw new Exception("Database connection is not available");
            }

            // Ensure the account_deletions table exists
            $this->ensureAccountDeletionsTableExists();
            
            // Start transaction
            $this->conn->beginTransaction();

            // First, log the deletion reason
            $logQuery = "INSERT INTO account_deletions (user_id, reason, deletion_date) 
                        VALUES (:user_id, :reason, NOW())";
            $logStmt = $this->conn->prepare($logQuery);
            $logStmt->execute([
                ':user_id' => $userId,
                ':reason' => $reason
            ]);

            // Update subscriber registration status to indicate deletion
            $updateQuery = "UPDATE subscribers SET sRegStatus = 3, sLastActivity = NOW() WHERE sId = :user_id";
            $updateStmt = $this->conn->prepare($updateQuery);
            $updateStmt->execute([':user_id' => $userId]);

            // If everything is successful, commit the transaction
            $this->conn->commit();
            return true;
        } catch (Exception $e) {
            // If there's an error, rollback the transaction if connection exists
            if ($this->conn instanceof PDO) {
                $this->conn->rollBack();
            }
            throw new Exception("Failed to delete account: " . $e->getMessage());
        }
    }
}
